<?php

namespace App\Controllers\Api;

use App\Controllers\BaseController;
use App\Models\UserModel;
use CodeIgniter\API\ResponseTrait;
use Razorpay\Api\Api;

class AuthController extends BaseController
{
    use ResponseTrait;

    public function __construct()
    {
        helper(['jwt', 'form']);
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
        header('Access-Control-Allow-Methods: POST, GET, OPTIONS, PUT, DELETE');
        
        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            exit;
        }
    }

    public function login()
    {
        helper('jwt'); 
        $response = service('response');
        
        // Only allow POST requests
        if ($this->request->getMethod() !== 'post') {
            return $response->setStatusCode(405)->setJSON([
                'status' => false,
                'message' => 'Method not allowed'
            ]);
        }

        // Get input data
        $contentType = $this->request->getHeaderLine('Content-Type');
        $isJson = strpos($contentType, 'application/json') !== false;
        
        if ($isJson) {
            $data = $this->request->getJSON(true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                return $response->setStatusCode(400)->setJSON([
                    'status' => false,
                    'message' => 'Invalid JSON format: ' . json_last_error_msg()
                ]);
            }
        } else {
            $data = $this->request->getPost();
        }

        // Validate required fields
        if (empty($data['email'])) {
            return $response->setStatusCode(400)->setJSON([
                'status' => false,
                'message' => 'Email is required'
            ]);
        }

        if (empty($data['password'])) {
            return $response->setStatusCode(400)->setJSON([
                'status' => false,
                'message' => 'Password is required'
            ]);
        }

        $email = $data['email'];
        $password = $data['password'];

        $model = new UserModel();
        $user = $model->where('email', $email)->orWhere('mobile', $email)->first();

        if (!$user) {
            return $response->setStatusCode(404)->setJSON([
                'status' => false,
                'message' => 'Account not found'
            ]);
        }

        if (!password_verify($password, $user['password'])) {
            return $response->setStatusCode(401)->setJSON([
                'status' => false,
                'message' => 'Invalid credentials'
            ]);
        }

        // Generate token
        $userData = [
            'id' => $user['id'],
            'name' => $user['name'],
            'email' => $user['email'],
            'user_type' => $user['user_type']
        ];

        $token = generate_jwt($userData);

        // Set session data
        $session = session();
        $session->set([
            'id' => $user['id'],
            'name' => $user['name'],
            'email' => $user['email'],
            'user_type' => $user['user_type'],
            'logged_in' => true
        ]);

        return $response->setJSON([
            'status' => true,
            'message' => 'Login successful',
            'token' => $token,
            'user' => $userData,
            'session' => [
                'id' => $session->get('id'),
                'logged_in' => $session->get('logged_in')
            ]
        ]);
    }
    
    public function register()
    {
        $response = service('response');
        $session = session();

        try {
            $contentType = $this->request->getHeaderLine('Content-Type');
            $isJson = strpos($contentType, 'application/json') !== false;
            
            $input = $isJson ? $this->request->getJSON(true) : $this->request->getPost();
            
            if ($input === null) {
                return $response->setStatusCode(400)->setJSON([
                    'status' => false,
                    'message' => 'Invalid request data format'
                ]);
            }

            $rules = [
                'name'     => 'required|min_length[3]|max_length[50]',
                'email'    => 'required|valid_email|is_unique[users.email]',
                'mobile'   => 'required|min_length[10]|max_length[10]|is_unique[users.mobile]',
                'password' => 'required|min_length[6]',
                'captcha'  => 'required'
            ];

            if (!$this->validate($rules)) {
                return $response->setStatusCode(422)->setJSON([
                    'status' => false,
                    'message' => 'Validation errors',
                    'errors' => $this->validator->getErrors()
                ]);
            }

            // Verify captcha (temporarily hardcoded for testing)
            $expectedCaptcha = 10; 
            if ($input['captcha'] != $expectedCaptcha) {
                return $response->setStatusCode(400)->setJSON([
                    'status' => false,
                    'message' => 'Invalid captcha answer'
                ]);
            }

            $this->deactivateExpiredUsers();

            $userData = [
                'name'       => $input['name'],
                'email'      => $input['email'],
                'mobile'     => $input['mobile'],
                'password'   => password_hash($input['password'], PASSWORD_DEFAULT),
                'referralId' => $input['referralId'] ?? null,
                'user_type'  => 'user',
                'status'     => 'inactive', 
                'created_at' => date('Y-m-d H:i:s')
            ];

            $userModel = new UserModel();
            if (!$userModel->save($userData)) {
                throw new \RuntimeException('Failed to save user');
            }

            $user = $userModel->select('id, name, email, mobile, user_type, created_at')
                             ->find($userModel->getInsertID());

            $this->sendWelcomeEmail($user, $input['password']);
             
            $loginResult = $this->autoLogin($input['email'], $input['password']);

            if (!$loginResult['success']) {
                throw new \RuntimeException('Auto-login failed after registration');
            }

            return $response->setJSON([
                'status'  => true,
                'message' => 'User registered and logged in successfully',
                'data'    => $loginResult['user'],
                'session' => [
                    'id' => session()->get('id'),
                    'logged_in' => true
                ]
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Registration Error: ' . $e->getMessage());
            return $response->setStatusCode(500)->setJSON([
                'status' => false,
                'message' => 'Internal server error',
                'error' => ENVIRONMENT === 'development' ? $e->getMessage() : null
            ]);
        }
    }

    private function autoLogin(string $email, string $password): array
    {
        $userModel = new UserModel();
        $user = $userModel->where('email', $email)->first();

        if (!$user || !password_verify($password, $user['password'])) {
            return ['success' => false, 'error' => 'Invalid credentials'];
        }

        session()->regenerate(); 
        $agent = $this->request->getUserAgent();

        $sessionData = [
            'id'         => $user['id'],
            'name'       => $user['name'],
            'email'      => $user['email'],
            'user_type'  => $user['user_type'],
            'logged_in'  => true,
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $agent->getAgentString(),
            'last_login' => date('Y-m-d H:i:s')
        ];

        session()->set($sessionData);

        $userModel->update($user['id'], ['last_login' => date('Y-m-d H:i:s')]);

        return [
            'success' => true,
            'user' => [
                'id'        => $user['id'],
                'name'      => $user['name'],
                'email'     => $user['email'],
                'user_type' => $user['user_type'],
                'mobile'    => $user['mobile'] ?? null
            ]
        ];
    }

    private function deactivateExpiredUsers()
    {
        $model = new UserModel();
        $oneYearAgo = date('Y-m-d', strtotime('-1 year'));
        
        $model->where('status', 'active')
              ->where('verifie_date <=', $oneYearAgo)
              ->set(['status' => 'inactive'])
              ->update();
    }

    private function sendWelcomeEmail($user, $plainPassword)
    {
        // Implement your email sending logic here
        // Example using CodeIgniter's email library:
        /*
        $email = \Config\Services::email();
        $email->setTo($user['email']);
        $email->setSubject('Welcome to Our Service');
        $email->setMessage(view('emails/welcome', [
            'user' => $user,
            'password' => $plainPassword
        ]));
        $email->send();
        */
    }

    public function user_success_payment($id)
    {
        $response = service('response');
        
        try {
            // Verify user exists
            $userModel = new UserModel();
            $user = $userModel->where('id', $id)->first();
            
            if (!$user) {
                return $response->setStatusCode(404)->setJSON([
                    'status' => false,
                    'message' => 'User not found'
                ]);
            }

            // Get Razorpay credentials from database
            $db = db_connect();
            $builder = $db->table('website_razor_transaction');
            $razorpayConfig = $builder->get()->getRowArray();
            
            if (!$razorpayConfig) {
                throw new \RuntimeException('Razorpay configuration not found');
            }

            // Initialize Razorpay API
            $api = new Api($razorpayConfig['merchantId'], $razorpayConfig['apiKey']);

            // Create order
            $amount = 100; // Fixed amount or get from configuration
            $orderData = [
                'amount'          => $amount * 100, // Convert to paise
                'currency'        => 'INR',
                'receipt'         => 'ORD_'.time().'_'.$id,
                'payment_capture' => 1 // Auto-capture payment
            ];

            $razorpayOrder = $api->order->create($orderData);

            // Prepare response data
            $responseData = [
                'user' => [
                    'id' => $user['id'],
                    'name' => $user['name'],
                    'email' => $user['email']
                ],
                'payment' => [
                    'amount' => $amount,
                    'currency' => 'INR',
                    'order_id' => $razorpayOrder->id,
                    'merchant_id' => $razorpayConfig['merchantId'],
                    'key' => $razorpayConfig['apiKey']
                ]
            ];

            return $response->setJSON([
                'status' => true,
                'message' => 'Payment order created successfully',
                'data' => $responseData
            ]);

        } catch (\Razorpay\Api\Errors\BadRequestError $e) {
            log_message('error', 'Razorpay API Error: ' . $e->getMessage());
            return $response->setStatusCode(400)->setJSON([
                'status' => false,
                'message' => 'Payment processing error',
                'error' => ENVIRONMENT === 'development' ? $e->getMessage() : null
            ]);
            
        } catch (\Exception $e) {
            log_message('error', 'Payment Error: ' . $e->getMessage());
            return $response->setStatusCode(500)->setJSON([
                'status' => false,
                'message' => 'Internal server error',
                'error' => ENVIRONMENT === 'development' ? $e->getMessage() : null
            ]);
        }
    }
}